<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User\EmailVerification;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Tools\OTPMailer;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\User\UserEnum;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Tables\User\UserEmailVerificationTable;
use Erlage\Photogram\Data\Models\User\EmailVerification\UserEmailVerificationEnum;
use Erlage\Photogram\Data\Models\User\EmailVerification\UserEmailVerificationModel;
use Erlage\Photogram\Data\Models\User\EmailVerification\UserEmailVerificationFinder;
use Erlage\Photogram\Data\Models\User\EmailVerification\UserEmailVerificationBuilder;

final class UserEmailVerificationActions extends ExceptionalRequests
{
    public static function start(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(
                UserTable::ID,
                RequestTable::PAYLOAD,
                UserTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | confirm that user is not already verified
            |--------------------------------------------------------------------------
            */

            if ($targetUserModel -> isEmailVerified())
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | whether generate a new reqeust or reuse previous one(if exists)
            |--------------------------------------------------------------------------
            */

            $generateNewRequest = true;

            /*
            |--------------------------------------------------------------------------
            | find the most recent request
            |--------------------------------------------------------------------------
            */

            $emailVerificationFinder = (new UserEmailVerificationFinder())
                -> setUserId($targetUserModel -> getId())
                -> setMetaIsExpired(UserEmailVerificationEnum::META_IS_EXPIRED_NO)
                -> limit('1')
                -> find();

            /*
            |--------------------------------------------------------------------------
            | if found make sure not expired
            |--------------------------------------------------------------------------
            */

            if ($emailVerificationFinder -> isFound())
            {
                $emailVerificationModel = $emailVerificationFinder -> popModelFromResults();

                if ($emailVerificationModel -> isObsolute())
                {
                    $emailVerificationModel -> update(
                        array(
                            UserEmailVerificationTable::META_IS_EXPIRED => UserEmailVerificationEnum::META_IS_EXPIRED_YES,
                        )
                    );

                    $emailVerificationModel -> save();
                }
                else
                {
                    $generateNewRequest = false;
                }
            }

            /*
            |--------------------------------------------------------------------------
            | finally generate new request if not found or expired
            |--------------------------------------------------------------------------
            */

            if ($generateNewRequest)
            {
                $emailVerificationModel = (new UserEmailVerificationBuilder())
                    -> setUserId($targetUserModel -> getId())
                    -> dispense();

                $emailVerificationModel -> save();
            }

            /*
            |--------------------------------------------------------------------------
            | send email to user's email
            |--------------------------------------------------------------------------
            */

            (new OTPMailer())
                -> setOTP($emailVerificationModel -> getMetaAccessOTP())
                -> setUserModel($targetUserModel)
                -> dispatchVerificationEmail();

            /*
            |--------------------------------------------------------------------------
            | create a dummy model containing only reference id(primary id) of request
            | which they can use while submitting otp in confirmation step
            |--------------------------------------------------------------------------
            */

            $refOnlyEmailVerificationModel = (new UserEmailVerificationBuilder())
                -> setUserId('_')
                -> setMetaAccessOTP('_')
                -> setStampLastUpdate('_')
                -> setStampRegistration('_')
                -> dispense();

            // add id of entry

            $refOnlyEmailVerificationModel -> update(
                array(
                    UserEmailVerificationTable::ID      => $emailVerificationModel -> getId(),
                    UserEmailVerificationTable::USER_ID => $targetUserModel -> getId(),
                )
            );

            /*
            |--------------------------------------------------------------------------
            | add dummy model to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserEmailVerificationTable::getTableName(), $refOnlyEmailVerificationModel -> getDataMap());
        });
    }

    public static function confirm(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $emailVerificationIdFromReq = self::$request -> findKey(
                UserEmailVerificationTable::ID,
                RequestTable::PAYLOAD,
                UserEmailVerificationTable::TABLE_NAME
            );

            $metaAccessOTPFromReq = self::$request -> findKey(
                UserEmailVerificationTable::META_ACCESS_OTP,
                RequestTable::PAYLOAD,
                UserEmailVerificationTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $emailVerificationIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target request exists
            |--------------------------------------------------------------------------
            */

            $targetEmailVerificationModel = UserEmailVerificationModel::findFromId_throwException($emailVerificationIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            /**
             * flag: not used anywhere else
             */
            $targetUserModel = UserModel::findFromId_throwException($targetEmailVerificationModel -> getUserId());

            /*
            |--------------------------------------------------------------------------
            | check request expiry
            |--------------------------------------------------------------------------
            */

            if ($targetEmailVerificationModel -> isExpired())
            {
                throw new RequestException(ResponseConstants::D_ERROR_OTP_EXPIRED);
            }

            if ($targetEmailVerificationModel -> isObsolute())
            {
                $targetEmailVerificationModel -> update(
                    array(
                        UserEmailVerificationTable::META_IS_EXPIRED => UserEmailVerificationEnum::META_IS_EXPIRED_YES,
                    )
                );

                $targetEmailVerificationModel -> save();

                throw new RequestException(ResponseConstants::D_ERROR_OTP_EXPIRED);
            }

            /*
            |--------------------------------------------------------------------------
            | confirm otp
            |--------------------------------------------------------------------------
            */

            if ($targetEmailVerificationModel -> getMetaAccessOTP() != $metaAccessOTPFromReq)
            {
                /*
                |--------------------------------------------------------------------------
                | increment attempt count and expire is it maxed out
                |--------------------------------------------------------------------------
                */

                $timesFailed = (string) (
                    (
                        (int) ($targetEmailVerificationModel -> getMetaTimesFailed())
                    ) + 1
                );

                $fieldsToUpdate = array(
                    UserEmailVerificationTable::META_TIMES_FAILED => $timesFailed,
                );

                if ((int) $timesFailed >= Settings::getInt(ServerConstants::SS_INT_OTP_LIFETIME_IN_TRIES))
                {
                    $fieldsToUpdate[UserEmailVerificationTable::META_IS_EXPIRED] = UserEmailVerificationEnum::META_IS_EXPIRED_YES;
                }

                $targetEmailVerificationModel -> update($fieldsToUpdate);

                $targetEmailVerificationModel -> save();

                /*
                |--------------------------------------------------------------------------
                | stop and tell client about incorrect otp
                |--------------------------------------------------------------------------
                */

                throw new RequestException(ResponseConstants::D_ERROR_OTP_MISMATCH);
            }

            /*
            |--------------------------------------------------------------------------
            | verification success, save changes
            |--------------------------------------------------------------------------
            */

            $targetUserModel -> update(
                array(
                    UserTable::META_IS_EMAIL_VERIFIED => UserEnum::META_IS_EMAIL_VERIFIED_YES,
                )
            );

            $targetUserModel -> save();

            /*
            |--------------------------------------------------------------------------
            | mark request as obsolute
            |--------------------------------------------------------------------------
            */

            $targetEmailVerificationModel -> update(
                array(
                    UserEmailVerificationTable::META_IS_EXPIRED => UserEmailVerificationEnum::META_IS_EXPIRED_YES,
                )
            );

            $targetEmailVerificationModel -> save();

            /*
            |--------------------------------------------------------------------------
            | send current user
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), $targetUserModel -> getDataMap());
        });
    }
}
